<?php
// guardianapi/update_password.php
declare(strict_types=1);

// Include config only if not already loaded
if (!function_exists('getDbConnection')) {
    require_once __DIR__ . '/config.php';
}

// Basic CORS using your helper if present
if (function_exists('gp_set_cors_all')) {
    gp_set_cors_all();
} else {
    header("Access-Control-Allow-Origin: *");
    header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
    header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
}

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

$respond = function(array $arr, int $code = 200): void {
    if (function_exists('gp_respond')) {
        gp_respond($arr, $code);
    } else {
        http_response_code($code);
        header('Content-Type: application/json; charset=UTF-8');
        echo json_encode($arr);
        exit;
    }
};

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $respond(["success"=>false,"status"=>"error","message"=>"Use POST"], 405);
}

// Parse JSON
$input = function_exists('gp_json_input')
    ? gp_json_input()
    : (function () {
        $raw = file_get_contents('php://input');
        $d = json_decode($raw, true);
        return is_array($d) ? $d : [];
    })();

if (!isset($input['user_id'], $input['new_password'])) {
    $respond(["success"=>false,"status"=>"error","message"=>"Invalid input"], 400);
}

$userId = (int)$input['user_id'];
$newPassword = (string)$input['new_password'];
if ($userId <= 0 || $newPassword === '') {
    $respond(["success"=>false,"status"=>"error","message"=>"Invalid user_id or password"], 400);
}

// Hash the password
$hash = password_hash($newPassword, PASSWORD_BCRYPT);
if ($hash === false) {
    $respond(["success"=>false,"status"=>"error","message"=>"Password hash failed"], 500);
}

$conn = getDbConnection();

/**
 * Try to detect the correct PK column name for guardusers:
 * 1) user_id
 * 2) id
 * Returns 'user_id' or 'id' (string), or null on failure.
 */
$detectPk = function(mysqli $conn): ?string {
    // Try SELECT with user_id
    $sql1 = "SELECT `user_id` FROM `users` WHERE `user_id` = ? LIMIT 1";
    $stmt1 = $conn->prepare($sql1);
    if ($stmt1) { $stmt1->close(); return 'user_id'; }

    // Try SELECT with id
    $sql2 = "SELECT `id` FROM `dusers` WHERE `id` = ? LIMIT 1";
    $stmt2 = $conn->prepare($sql2);
    if ($stmt2) { $stmt2->close(); return 'id'; }

    // Both failed -> table missing or other schema issue
    error_log("[update_password] Could not prepare SELECT for PK detection. Errors: "
        . "user_id SQL error={$conn->error}");
    return null;
};

$pk = $detectPk($conn);
if ($pk === null) {
    // Don’t leak schema specifics in API response; log details already done above
    $respond([
        "success"=>false,
        "status"=>"error",
        "message"=>"Database schema mismatch for users (see server logs)"
    ], 500);
}

/**
 * Verify user exists (optional but gives better 404)
 */
$checkSql = "SELECT `$pk` FROM `users` WHERE `$pk` = ? LIMIT 1";
$checkStmt = $conn->prepare($checkSql);
if (!$checkStmt) {
    error_log("[update_password] Prepare failed (check user): SQL={$checkSql} ERR={$conn->error}");
    $respond(["success"=>false,"status"=>"error","message"=>"Database error (check)"], 500);
}
$checkStmt->bind_param("i", $userId);
$checkStmt->execute();
$checkStmt->store_result();
if ($checkStmt->num_rows === 0) {
    $checkStmt->close();
    $respond(["success"=>false,"status"=>"error","message"=>"User not found"], 404);
}
$checkStmt->close();

/**
 * Try UPDATE with updated_at column; if prepare fails, fall back to without updated_at.
 */
$updated = false;
$usedFallback = false;

// First attempt: with updated_at
$updateSql1 = "UPDATE `users` SET `password` = ?, `updated_at` = NOW() WHERE `$pk` = ?";
$stmt = $conn->prepare($updateSql1);
if ($stmt) {
    $stmt->bind_param("si", $hash, $userId);
    $ok = $stmt->execute();
    $affected = $stmt->affected_rows ?? 0;
    $stmt->close();
    $updated = $ok && $affected > 0;
} else {
    // Log and try fallback
    error_log("[update_password] Prepare failed (with updated_at): SQL={$updateSql1} ERR={$conn->error}");
    $usedFallback = true;

    $updateSql2 = "UPDATE `users` SET `password` = ? WHERE `$pk` = ?";
    $stmt2 = $conn->prepare($updateSql2);
    if (!$stmt2) {
        error_log("[update_password] Prepare failed (fallback without updated_at): SQL={$updateSql2} ERR={$conn->error}");
        $respond(["success"=>false,"status"=>"error","message"=>"Database error (update)"], 500);
    }
    $stmt2->bind_param("si", $hash, $userId);
    $ok2 = $stmt2->execute();
    $affected2 = $stmt2->affected_rows ?? 0;
    $stmt2->close();
    $updated = $ok2 && $affected2 > 0;
}

if ($updated) {
    $respond([
        "success"=>true,
        "status"=>"success",
        "message"=>"Password updated successfully",
        "meta"=>["fallback_without_updated_at"=>$usedFallback]
    ]);
}

// UPDATE ran but didn’t change a row — could be same hash or row locked
$respond([
    "success"=>false,
    "status"=>"error",
    "message"=>"No update performed (unchanged or restricted)",
    "meta"=>["fallback_without_updated_at"=>$usedFallback]
], 400);
