<?php
// guardianapi/sms_quota_get.php
declare(strict_types=1);

// Auth is enforced by index.php (via $GLOBALS['AUTH'])
if (!isset($GLOBALS['AUTH'])) { http_response_code(401); echo json_encode(['status'=>'error','message'=>'Unauthorized']); exit; }

require_once __DIR__ . '/config.php';
$conn = getDbConnection();

if (!function_exists('out')) {
  function out($arr, int $code=200){ http_response_code($code); echo json_encode($arr); exit; }
}

/** ---------- Helpers ---------- */
function iifset($arr, $key, $def=null){ return isset($arr[$key]) ? $arr[$key] : $def; }

/** map: org -> used_day / used_month */
function usage_map_by_org(mysqli $c): array {
  $maps = ['day'=>[], 'month'=>[]];

  $sqlDay = "
    SELECT organisation_id, COUNT(*) n
      FROM sms_messages
     WHERE status IN ('queued','sent')
       AND (
         (sent_at IS NOT NULL AND DATE(sent_at)=CURDATE())
         OR (sent_at IS NULL AND DATE(NOW())=CURDATE())
       )
     GROUP BY organisation_id";
  if ($res = $c->query($sqlDay)) {
    while ($r = $res->fetch_assoc()) $maps['day'][(int)$r['organisation_id']] = (int)$r['n'];
    $res->close();
  }

  $sqlMon = "
    SELECT organisation_id, COUNT(*) n
      FROM sms_messages
     WHERE status IN ('queued','sent')
       AND (
         (sent_at IS NOT NULL AND DATE_FORMAT(sent_at,'%Y-%m')=DATE_FORMAT(CURDATE(),'%Y-%m'))
         OR (sent_at IS NULL AND DATE_FORMAT(NOW(),'%Y-%m')=DATE_FORMAT(CURDATE(),'%Y-%m'))
       )
     GROUP BY organisation_id";
  if ($res = $c->query($sqlMon)) {
    while ($r = $res->fetch_assoc()) $maps['month'][(int)$r['organisation_id']] = (int)$r['n'];
    $res->close();
  }

  return $maps;
}

/** map: house -> used_day / used_month */
function usage_map_by_house(mysqli $c, ?int $orgId=null): array {
  $maps = ['day'=>[], 'month'=>[]];
  $whereOrg = $orgId ? " AND organisation_id=".(int)$orgId : "";

  $sqlDay = "
    SELECT house_id, COUNT(*) n
      FROM sms_messages
     WHERE status IN ('queued','sent')
       AND (
         (sent_at IS NOT NULL AND DATE(sent_at)=CURDATE())
         OR (sent_at IS NULL AND DATE(NOW())=CURDATE())
       )
       AND house_id IS NOT NULL{$whereOrg}
     GROUP BY house_id";
  if ($res = $c->query($sqlDay)) {
    while ($r = $res->fetch_assoc()) $maps['day'][(int)$r['house_id']] = (int)$r['n'];
    $res->close();
  }

  $sqlMon = "
    SELECT house_id, COUNT(*) n
      FROM sms_messages
     WHERE status IN ('queued','sent')
       AND (
         (sent_at IS NOT NULL AND DATE_FORMAT(sent_at,'%Y-%m')=DATE_FORMAT(CURDATE(),'%Y-%m'))
         OR (sent_at IS NULL AND DATE_FORMAT(NOW(),'%Y-%m')=DATE_FORMAT(CURDATE(),'%Y-%m'))
       )
       AND house_id IS NOT NULL{$whereOrg}
     GROUP BY house_id";
  if ($res = $c->query($sqlMon)) {
    while ($r = $res->fetch_assoc()) $maps['month'][(int)$r['house_id']] = (int)$r['n'];
    $res->close();
  }

  return $maps;
}

/** fetch org-level limits (house_id IS NULL) keyed by org */
function org_quota_rows(mysqli $c, ?int $orgId=null): array {
  $rows = [];
  $sql = "SELECT organisation_id,
                 MAX(CASE WHEN period='daily'   THEN limit_count END) AS per_day,
                 MAX(CASE WHEN period='monthly' THEN limit_count END) AS per_month
            FROM sms_quotas
           WHERE is_active=1 AND house_id IS NULL".
           ($orgId ? " AND organisation_id=?" : "").
         " GROUP BY organisation_id";
  if ($orgId) { $stmt = $c->prepare($sql); $stmt->bind_param("i", $orgId); $stmt->execute(); $res = $stmt->get_result(); }
  else { $res = $c->query($sql); }
  if ($res) {
    while ($r = $res->fetch_assoc()) {
      $oid = (int)$r['organisation_id'];
      $rows[$oid] = [
        'organisation_id' => $oid,
        'per_day'   => isset($r['per_day'])   ? (int)$r['per_day']   : null,
        'per_month' => isset($r['per_month']) ? (int)$r['per_month'] : null,
      ];
    }
    $res->close();
  }
  if ($orgId) $stmt->close();
  return $rows;
}

/** fetch house-level limits keyed by house */
function house_quota_rows(mysqli $c, ?int $orgId=null): array {
  $rows = [];
  $sql = "SELECT organisation_id, house_id,
                 MAX(CASE WHEN period='daily'   THEN limit_count END) AS per_day,
                 MAX(CASE WHEN period='monthly' THEN limit_count END) AS per_month
            FROM sms_quotas
           WHERE is_active=1 AND house_id IS NOT NULL".
           ($orgId ? " AND organisation_id=?" : "").
         " GROUP BY organisation_id, house_id";
  if ($orgId) { $stmt = $c->prepare($sql); $stmt->bind_param("i", $orgId); $stmt->execute(); $res = $stmt->get_result(); }
  else { $res = $c->query($sql); }
  if ($res) {
    while ($r = $res->fetch_assoc()) {
      $hid = (int)$r['house_id'];
      $rows[$hid] = [
        'organisation_id' => (int)$r['organisation_id'],
        'house_id' => $hid,
        'per_day'   => isset($r['per_day'])   ? (int)$r['per_day']   : null,
        'per_month' => isset($r['per_month']) ? (int)$r['per_month'] : null,
      ];
    }
    $res->close();
  }
  if ($orgId) $stmt->close();
  return $rows;
}

/** house id -> house_number */
function house_names(mysqli $c, array $houseIds): array {
  if (!$houseIds) return [];
  $ids = implode(',', array_map('intval', $houseIds));
  $map = [];
  $sql = "SELECT id, house_number FROM houses WHERE id IN ($ids)";
  if ($res = $c->query($sql)) {
    while ($r = $res->fetch_assoc()) $map[(int)$r['id']] = (string)$r['house_number'];
    $res->close();
  }
  return $map;
}

/** compute pct helper */
function pct(?int $used, ?int $limit): ?float {
  if ($limit === null || $limit <= 0) return null;
  return round(($used / $limit) * 100, 2);
}

/** ---------- Main ---------- */
$auth  = $GLOBALS['AUTH'];
$role  = (string)($auth['role'] ?? '');
$actorOrg = (int)($auth['organisation_id'] ?? 0);

// Filters (all optional)
$orgId   = isset($_GET['organisation_id']) && $_GET['organisation_id'] !== '' ? (int)$_GET['organisation_id'] : null;
$houseId = isset($_GET['house_id'])        && $_GET['house_id'] !== ''        ? (int)$_GET['house_id']        : null;

// Default scoping
if ($orgId === null) {
  if ($role === 'admin' && ($actorOrg === 0 || $actorOrg === 7 || (defined('SUPERADMIN_ORG_ID') && $actorOrg===(int)SUPERADMIN_ORG_ID))) {
    // keep null -> global
  } else {
    $orgId = $actorOrg;
  }
}
if (!($role === 'admin' && ($actorOrg===0 || $actorOrg===7 || (defined('SUPERADMIN_ORG_ID') && $actorOrg===(int)SUPERADMIN_ORG_ID))) && $orgId !== null && $orgId !== $actorOrg) {
  $orgId = $actorOrg;
}

// Build response
$resp = [
  'status' => 'success',
  'scope'  => [
    'mode' => ($orgId === null ? 'global' : ($houseId ? 'house' : 'org')),
    'organisation_id' => $orgId,
    'house_id' => $houseId,
  ],
];

// GLOBAL view (superadmin, no org filter)
if ($orgId === null) {
  $orgLimits   = org_quota_rows($conn, null);
  $houseLimits = house_quota_rows($conn, null);
  $uOrg        = usage_map_by_org($conn);
  $uHouse      = usage_map_by_house($conn, null);

  $orgList = [];
  foreach ($orgLimits as $oid => $q) {
    $usedDay   = (int) iifset($uOrg['day'], $oid, 0);
    $usedMonth = (int) iifset($uOrg['month'], $oid, 0);
    $orgList[] = [
      'organisation_id' => $oid,
      'per_day'         => $q['per_day'],
      'per_month'       => $q['per_month'],
      'used_day'        => $usedDay,
      'used_month'      => $usedMonth,
      'pct_day'         => pct($usedDay, $q['per_day']),
      'pct_month'       => pct($usedMonth, $q['per_month']),
      'monthly_limit'   => $q['per_month'],
      'sent_this_month' => $usedMonth,
    ];
  }

  $houseIds  = array_keys($houseLimits);
  $nameMap   = house_names($conn, $houseIds);
  $houseList = [];
  foreach ($houseLimits as $hid => $q) {
    $usedDay   = (int) iifset($uHouse['day'], $hid, 0);
    $usedMonth = (int) iifset($uHouse['month'], $hid, 0);
    $houseList[] = [
      'organisation_id' => (int)$q['organisation_id'],
      'house_id'        => (int)$hid,
      'house_number'    => iifset($nameMap, $hid, null),
      'per_day'         => $q['per_day'],
      'per_month'       => $q['per_month'],
      'used_day'        => $usedDay,
      'used_month'      => $usedMonth,
      'pct_day'         => pct($usedDay, $q['per_day']),
      'pct_month'       => pct($usedMonth, $q['per_month']),
      'monthly_limit'   => $q['per_month'],
      'sent_this_month' => $usedMonth,
    ];
  }

  $resp['org_quotas']   = array_values($orgList);
  $resp['house_quotas'] = array_values($houseList);
}

/** ORG or HOUSE scoped view */
$limitsOrg   = org_quota_rows($conn, $orgId);
$limitsHouse = house_quota_rows($conn, $orgId);
$uOrg        = usage_map_by_org($conn);
$uHouse      = usage_map_by_house($conn, $orgId);

// House summary
if ($houseId) {
  $hq = iifset($limitsHouse, $houseId, ['per_day'=>null,'per_month'=>null, 'organisation_id'=>$orgId, 'house_id'=>$houseId]);
  $usedDay   = (int) iifset($uHouse['day'], $houseId, 0);
  $usedMonth = (int) iifset($uHouse['month'], $houseId, 0);

  $resp['per_day']    = $hq['per_day'];
  $resp['per_month']  = $hq['per_month'];
  $resp['used_day']   = $usedDay;
  $resp['used_month'] = $usedMonth;
  $resp['pct_day']    = pct($usedDay, $hq['per_day']);
  $resp['pct_month']  = pct($usedMonth, $hq['per_month']);

  out($resp);
}

// ORG summary
$oq = iifset($limitsOrg, $orgId, ['per_day'=>null,'per_month'=>null]);
$usedDayOrg   = (int) iifset($uOrg['day'], $orgId, 0);
$usedMonthOrg = (int) iifset($uOrg['month'], $orgId, 0);

$resp['per_day']    = $oq['per_day'];
$resp['per_month']  = $oq['per_month'];
$resp['used_day']   = $usedDayOrg;
$resp['used_month'] = $usedMonthOrg;
$resp['pct_day']    = pct($usedDayOrg, $oq['per_day']);
$resp['pct_month']  = pct($usedMonthOrg, $oq['per_month']);

// Include each house quota under this org
$houseIds  = array_keys($limitsHouse);
$nameMap   = house_names($conn, $houseIds);
$houseList = [];
foreach ($limitsHouse as $hid => $q) {
  $ud = (int) iifset($uHouse['day'], $hid, 0);
  $um = (int) iifset($uHouse['month'], $hid, 0);
  $houseList[] = [
    'organisation_id' => (int)$orgId,
    'house_id'        => (int)$hid,
    'house_number'    => iifset($nameMap, $hid, null),
    'per_day'         => $q['per_day'],
    'per_month'       => $q['per_month'],
    'used_day'        => $ud,
    'used_month'      => $um,
    'pct_day'         => pct($ud, $q['per_day']),
    'pct_month'       => pct($um, $q['per_month']),
    'monthly_limit'   => $q['per_month'],
    'sent_this_month' => $um,
  ];
}
$resp['house_quotas'] = array_values($houseList);

out($resp);
