<?php
// guardianapi/sms_get_dlr.php
declare(strict_types=1);

if (!isset($GLOBALS['AUTH'])) { http_response_code(401); echo json_encode(['status'=>'error','message'=>'Unauthorized']); exit; }
$auth = $GLOBALS['AUTH'];

require_once __DIR__ . '/config.php';
$conn = getDbConnection();

function jerr($msg, $code=400){ http_response_code($code); echo json_encode(['status'=>'error','message'=>$msg]); exit; }

// ---- helpers ----
function provider_get_dlr(string $providerMsgId): array {
  $payload = [
    'apikey'    => SMS_APIKEY,
    'partnerID' => SMS_PARTNER,
    'messageID' => $providerMsgId,
  ];
  $ch = curl_init(SMS_BASE . 'getdlr/');
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
    CURLOPT_POST           => true,
    CURLOPT_POSTFIELDS     => json_encode($payload),
    CURLOPT_TIMEOUT        => 30,
  ]);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($err) {
    return ['ok'=>false,'status_code'=>0,'status_desc'=>'transport error: '.$err,'raw'=>$body?:''];
  }

  $data = json_decode($body, true);
  // Normalise fields defensively (provider formats vary)
  $statusCode = (int)($data['status'] ?? $data['code'] ?? $data['status_code'] ?? 0);
  $statusDesc = (string)($data['status_desc'] ?? $data['description'] ?? $data['message'] ?? '');

  // Some providers return array items
  if (!$statusCode && is_array($data)) {
    $first = $data[0] ?? null;
    if (is_array($first)) {
      $statusCode = (int)($first['status'] ?? $first['code'] ?? 0);
      $statusDesc = (string)($first['status_desc'] ?? $first['description'] ?? '');
    }
  }

  return ['ok'=>($code>=200 && $code<500), 'status_code'=>$statusCode, 'status_desc'=>$statusDesc, 'raw'=>$body?:''];
}

function map_final_status($statusCode, $statusDesc): ?string {
  $desc = strtoupper(trim((string)$statusDesc));
  if ($statusCode === 200 || $statusCode === 1 || $desc === 'DELIVRD' || strpos($desc, 'DELIVER') !== false) {
    return 'delivered';
  }
  if ($statusCode >= 400 || in_array($statusCode, [1005,1006,500,503], true) || strpos($desc, 'FAILED') !== false) {
    return 'failed';
  }
  return null; // still pending / unknown
}

function store_dlr(mysqli $conn, ?int $smsId, int $orgId, string $providerMsgId, int $statusCode, string $statusDesc, string $raw): void {
  $sql = "INSERT INTO sms_delivery_reports (sms_id, organisation_id, provider_message_id, status_code, status_desc, raw_json)
          VALUES (?,?,?,?,?,?)";
  $stmt = $conn->prepare($sql);
  $stmt->bind_param("iisisb", $smsId, $orgId, $providerMsgId, $statusCode, $statusDesc, $raw);
  // send_long_data for the blob param (index 5 zero-based -> 5)
  $stmt->send_long_data(5, $raw);
  $stmt->execute();
  $stmt->close();
}

function update_master_row(mysqli $conn, int $smsId, ?string $final): void {
  if (!$final) return;
  if ($final === 'delivered') {
    $sql = "UPDATE sms_messages SET status='delivered', delivered_at=NOW() WHERE id=?";
  } else if ($final === 'failed') {
    $sql = "UPDATE sms_messages SET status='failed' WHERE id=?";
  } else { return; }
  $stmt = $conn->prepare($sql);
  $stmt->bind_param("i", $smsId);
  $stmt->execute();
  $stmt->close();
}

function get_row_by_sms_id(mysqli $conn, int $smsId, int $orgId): ?array {
  $stmt = $conn->prepare("SELECT id, provider_message_id FROM sms_messages WHERE id=? AND organisation_id=? LIMIT 1");
  $stmt->bind_param("ii", $smsId, $orgId);
  $stmt->execute();
  $res = $stmt->get_result();
  $row = $res->fetch_assoc() ?: null;
  $stmt->close();
  return $row;
}

// ---- inputs ----
$input = json_decode(file_get_contents('php://input'), true) ?: [];
$orgId = (int)($input['organisation_id'] ?? ($auth['organisation_id'] ?? 0));
$messageId  = isset($input['messageid']) ? trim((string)$input['messageid']) : '';
$smsIdInput = isset($input['sms_id']) ? (int)$input['sms_id'] : 0;
$messageIds = isset($input['messageids']) && is_array($input['messageids']) ? $input['messageids'] : [];
$smsIds     = isset($input['sms_ids']) && is_array($input['sms_ids']) ? array_map('intval',$input['sms_ids']) : [];
$onlyPending= isset($input['only_pending']) ? (bool)$input['only_pending'] : true;
$limit      = isset($input['limit']) ? max(1, (int)$input['limit']) : 100;

// single by messageid or sms_id
$targets = []; // each: ['sms_id'=>int|null,'provider_message_id'=>string]
if ($messageId !== '') {
  $targets[] = ['sms_id'=>null,'provider_message_id'=>$messageId];
}
if ($smsIdInput > 0) {
  if ($orgId<=0) jerr('organisation_id required with sms_id');
  $row = get_row_by_sms_id($conn, $smsIdInput, $orgId);
  if (!$row || empty($row['provider_message_id'])) jerr('sms_id not found or missing provider_message_id',404);
  $targets[] = ['sms_id'=>(int)$row['id'], 'provider_message_id'=>$row['provider_message_id']];
}
// batch arrays (messageids / sms_ids)
foreach ($messageIds as $mid) {
  $mid = trim((string)$mid); if ($mid!=='') $targets[] = ['sms_id'=>null,'provider_message_id'=>$mid];
}
if (!empty($smsIds)) {
  if ($orgId<=0) jerr('organisation_id required with sms_ids array');
  $in = implode(',', array_fill(0, count($smsIds), '?'));
  $types = str_repeat('i', count($smsIds)) . 'i'; // smsIds + orgId
  $sql = "SELECT id, provider_message_id FROM sms_messages WHERE id IN ($in) AND organisation_id=?";
  $stmt = $conn->prepare($sql);
  $bind = [];
  $bind[] = & $types;
  foreach ($smsIds as $k=>$v) { $bind[] = & $smsIds[$k]; }
  $bind[] = & $orgId;
  call_user_func_array([$stmt, 'bind_param'], $bind);
  $stmt->execute();
  $res = $stmt->get_result();
  while ($row = $res->fetch_assoc()) {
    if (!empty($row['provider_message_id'])) {
      $targets[] = ['sms_id'=>(int)$row['id'], 'provider_message_id'=>$row['provider_message_id']];
    }
  }
  $stmt->close();
}

// batch by organisation (pending or all)
if (empty($targets) && $orgId>0) {
  $sql = "SELECT id, provider_message_id FROM sms_messages
          WHERE organisation_id=? AND provider_message_id IS NOT NULL";
  if ($onlyPending) $sql .= " AND status IN ('queued','sent')";
  $sql .= " ORDER BY id DESC LIMIT ?";
  $stmt = $conn->prepare($sql);
  $stmt->bind_param("ii", $orgId, $limit);
  $stmt->execute();
  $res = $stmt->get_result();
  while ($row = $res->fetch_assoc()) {
    $targets[] = ['sms_id'=>(int)$row['id'], 'provider_message_id'=>$row['provider_message_id']];
  }
  $stmt->close();
}

if (empty($targets)) jerr('No targets to query. Provide messageid / sms_id / arrays or organisation_id batch.', 400);

// ---- process ----
$results = [];
$errors  = [];
foreach ($targets as $t) {
  $provId = (string)$t['provider_message_id'];
  $sid    = isset($t['sms_id']) ? (int)$t['sms_id'] : null;

  $dlr = provider_get_dlr($provId);
  if (!$dlr['ok']) {
    $errors[] = ['provider_messageid'=>$provId,'sms_id'=>$sid,'error'=>$dlr['status_desc']];
    // still store what we got
  }

  $code = (int)($dlr['status_code'] ?? 0);
  $desc = (string)($dlr['status_desc'] ?? '');
  $raw  = (string)($dlr['raw'] ?? '');

  // if we don't have sms_id yet, try to look it up for storage linkage
  if ($sid === null && $orgId>0) {
    $stmt = $conn->prepare("SELECT id FROM sms_messages WHERE provider_message_id=? AND organisation_id=? LIMIT 1");
    $stmt->bind_param("si", $provId, $orgId);
    $stmt->execute(); $res=$stmt->get_result();
    if ($row=$res->fetch_assoc()) $sid=(int)$row['id'];
    $stmt->close();
  }

  store_dlr($conn, $sid, $orgId, $provId, $code, $desc, $raw);

  if ($sid !== null) {
    $final = map_final_status($code, $desc);
    update_master_row($conn, $sid, $final);
  }

  $results[] = [
    'provider_messageid' => $provId,
    'sms_id'             => $sid,
    'status_code'        => $code,
    'status_desc'        => $desc,
    'final'              => map_final_status($code, $desc)
  ];
}

echo json_encode([
  'status'   => 'success',
  'queried'  => count($targets),
  'results'  => $results,
  'errors'   => $errors
]);
