<?php
// guardianapi/index.php
declare(strict_types=1);

header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit(); }

require_once __DIR__ . '/config.php';
$conn = getDbConnection();

function out($arr, int $code=200){ http_response_code($code); echo json_encode($arr); exit(); }

/** Return route like:
 *  - ?r=visitor_get
 *  - PATH_INFO=/visitor_get
 *  - /guardianapi/visitor/create  -> "visitor/create"
 */
function route_path(): string {
  if (!empty($_GET['r'])) return trim((string)$_GET['r'], "/");
  $pi = $_SERVER['PATH_INFO'] ?? '';
  if ($pi) return trim($pi, "/");
  $uri = parse_url($_SERVER['REQUEST_URI'] ?? '', PHP_URL_PATH) ?? '';
  // if pretty URLs are enabled (/guardianapi/xyz)
  $base = rtrim(dirname($_SERVER['SCRIPT_NAME'] ?? ''), '/'); // /guestpassmultitenancy/guardianapi
  if (strpos($uri, $base) === 0) $uri = substr($uri, strlen($base));
  $uri = trim($uri, '/');
  if ($uri === 'index.php' || $uri === '') return '';
  // if they hit /guardianapi/index.php?r=... the code above handled it
  return preg_replace('/\.php$/i', '', $uri);
}

function bearer_token(): ?string {
  $h = null;
  if (function_exists('getallheaders')) {
    $all = getallheaders();
    $h = $all['Authorization'] ?? $all['authorization'] ?? null;
  }
  if (!$h && isset($_SERVER['HTTP_AUTHORIZATION'])) $h = $_SERVER['HTTP_AUTHORIZATION'];
  if ($h && preg_match('/Bearer\s+([A-Za-z0-9._\-]+)/', $h, $m)) return $m[1];
  // allow token via query for legacy clients
  if (!empty($_GET['token'])) return (string)$_GET['token'];
  return null;
}

function verify_token(mysqli $conn, ?string $token): ?array {
  if (!$token) return null;

  // admin
  if ($stmt = $conn->prepare("SELECT admin_id AS id, organisation_id FROM admin WHERE token=? AND is_active=1 LIMIT 1")) {
    $stmt->bind_param("s", $token);
    $stmt->execute(); $res = $stmt->get_result();
    if ($row = $res->fetch_assoc()) { $stmt->close(); return ['role'=>'admin','id'=>(int)$row['id'],'organisation_id'=>(int)$row['organisation_id'],'token'=>$token]; }
    $stmt->close();
  }
  // user (host/guard)
  if ($stmt = $conn->prepare("SELECT user_id AS id, organisation_id, house_id FROM users WHERE token=? LIMIT 1")) {
    $stmt->bind_param("s", $token);
    $stmt->execute(); $res = $stmt->get_result();
    if ($row = $res->fetch_assoc()) {
      $stmt->close();
      return ['role'=>'user','id'=>(int)$row['id'],'organisation_id'=>(int)$row['organisation_id'],'house_id'=>isset($row['house_id'])?(int)$row['house_id']:null,'token'=>$token];
    }
    $stmt->close();
  }
  return null;
}

/** Map of routes → handler PHP file (relative to this folder) */
$routes = [
  // org & auth
  'registerorg'                 => 'registerorg.php',
  'validate'                    => 'validate.php',
  'admin_signup'                => 'admin_signup.php',
  'admin_login'                 => 'admin_login.php',
  'user_login'                  => 'user_login.php',
  'update_admin_password'       => 'update_admin_password.php',

  // users
  'get_next_user_id'            => 'get_next_user_id.php',
  'register_user'               => 'register_user.php',
  'fetch_users'                 => 'fetch_users.php',
  'update_password'             => 'update_password.php',
  'userdatafetch'               => 'userdatafetch.php',

  // logs
  'savelog'                     => 'savelog.php',

  // houses
  'house_get'                   => 'house_get.php',
  'house_post'                  => 'house_post.php',

  // visitors
  'visitor_get'                 => 'visitor_get.php',
  'visitor_unexited_get'        => 'visitor_unexited_get.php',
  'visitor_post'                => 'visitor_post.php',
  'visitor_search'              => 'visitor_search.php',
  'visitor_update_status'       => 'visitor_update_status.php',
  'validate_otp'                => 'validate_otp.php',
  'batch_mark_visitor_exited'   => 'batch_mark_visitor_exited.php',

  // analytics
  'analytics_stats_live'        => 'analytics_stats_live.php',
  'visitor_analytics'           => 'visitor_analytics.php',

  // uploads
  'visitorphoto'                => 'visitorphoto.php',
  'uploadVisitorImage'          => 'uploadVisitorImage.php',

  // mailers
  'send_visitor_alert'          => 'send_visitor_alert.php',
  'contact_send'                => 'contact_send.php',
  // sms
  'sms_send'                   => 'sms_send.php',
  'sms_get_dlr'               => 'sms_get_dlr.php',
  // optional listing for admin usage reports
  'sms_log_get'               => 'sms_log_get.php',
   'sms_stats'      => 'sms_stats.php',
    'sms_quota_get'               => 'sms_quota_get.php',
    'sms_quota_set'               => 'sms_quota_set.php',
    'sms_balance'=>'sms_balance.php',


  // REST-style stubs (map to your own handlers)
  'visitor/create'              => 'visitor_create.php',
  'visit/create'                => 'visit_create.php',
  //Organisations
    'org_get'    => 'org_get.php',
    'org_post'   => 'org_post.php',
    'org_update' => 'org_update.php',
];

$public = [
  // open by design (Android currently calls these without auth headers)
  'registerorg','validate','admin_signup','admin_login','user_login',
  'get_next_user_id','register_user','fetch_users','update_password',
  'house_get','visitor_get','visitor_unexited_get','visitor_post','visitor_search',
  'analytics_stats_live','visitor_analytics','batch_mark_visitor_exited',
  'send_visitor_alert','contact_send',
  // allow password change to pass token in body (handler should verify)
  'update_admin_password',
];

$r = route_path();
if ($r === '') {
  out(['status'=>'ok','message'=>'GuestPass API index','endpoints'=>array_keys($routes)]);
}
if (!isset($routes[$r])) {
  // also try lowercased key
  $lk = strtolower($r);
  $found = null;
  foreach ($routes as $k=>$v) { if (strtolower($k)===$lk) { $found=$k; break; } }
  if ($found===null) out(['status'=>'error','message'=>'Unknown endpoint: '.$r],404);
  $r = $found;
}

if (!in_array($r, $public, true)) {
  $token = bearer_token();
  $auth  = verify_token($conn, $token);
  if (!$auth) out(['status'=>'error','message'=>'Unauthorized'],401);
  $GLOBALS['AUTH'] = $auth;
}

$target = __DIR__ . '/' . $routes[$r];
if (!is_file($target)) out(['status'=>'error','message'=>'Handler missing: '.$routes[$r]],500);

require $target;
exit;
