<?php
// visitor_post.php — OTP optional + error-only logging
header("Content-Type: application/json; charset=utf-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Best-effort: send PHP errors to local error.log (won't break if not permitted)
$logDir = __DIR__;
$logFile = $logDir . '/error.log';
if (is_dir($logDir) && is_writable($logDir)) {
    @ini_set('log_errors', '1');
    @ini_set('error_log', $logFile);
}

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

require_once __DIR__ . '/config.php';

function json_error(string $msg, int $code = 400) {
    // Log errors only
    error_log("visitor_post error ($code): " . $msg);
    http_response_code($code);
    echo json_encode(["status" => "error", "message" => $msg], JSON_UNESCAPED_UNICODE);
    exit;
}

try {
    $raw = file_get_contents('php://input');
    if (!$raw) json_error("Empty request body.", 400);

    $data = json_decode($raw, true);
    if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
        json_error("Invalid JSON.", 400);
    }

    // Match Android VisitorRecordRequest field names
    $organisation_id = isset($data['organisation_id']) ? (int)$data['organisation_id'] : null;
    $house_id        = isset($data['house_id'])        ? (int)$data['house_id']        : null;
    $description     = isset($data['description'])     ? trim((string)$data['description']) : '';
    $name            = isset($data['name'])            ? trim((string)$data['name'])        : '';
    $phone_number    = isset($data['phone_number'])    ? trim((string)$data['phone_number']) : '';
    $image_metadata  = isset($data['image_metadata'])  ? trim((string)$data['image_metadata']) : '';
    $id_plates       = isset($data['id_plates']) && $data['id_plates'] !== '' ? trim((string)$data['id_plates']) : null;
    // OTP is now OPTIONAL; store empty string if not provided (safer if column is NOT NULL)
    $otp_code        = isset($data['otp_code']) ? trim((string)$data['otp_code']) : '';
    $admitted_by     = isset($data['admitted_by'])     ? trim((string)$data['admitted_by'])  : '';

    // Minimal validation (OTP NOT required)
    if (!$house_id || $description === '' || $name === '' || $phone_number === '' || $admitted_by === '') {
        json_error("Missing required fields.", 400);
    }

    // Server-side safe fallback for NOT NULL column
    if ($image_metadata === '') $image_metadata = 'noimg'; // app also uses "noimg" when empty

    $db = getDbConnection();
    if (!$db) json_error("Database connection failed.", 500);
    $db->set_charset('utf8mb4');

    $sql = "
        INSERT INTO visitor_records
            (house_id, organisation_id, description, name, phone_number, image_metadata, id_plates, admitted_by, otp_code, is_exited, status)
        VALUES
            (?,        ?,               ?,           ?,    ?,            ?,              ?,         ?,           ?,        0,        'active')
    ";
    $st = $db->prepare($sql);
    if (!$st) json_error("Database error (prepare): " . $db->error, 500);

    // Exact match between 9 vars and 9 type chars
    $st->bind_param(
        "iisssssss",
        $house_id,
        $organisation_id,  // nullable ok
        $description,
        $name,
        $phone_number,
        $image_metadata,
        $id_plates,        // nullable ok
        $admitted_by,
        $otp_code          // now optional; empty string if not provided
    );

    if (!$st->execute()) {
        $err = $st->error ?: "Failed to save visitor record";
        $st->close(); $db->close();
        json_error($err, 500);
    }

    $insertId = $st->insert_id;
    $st->close();

    // Optional lookup retained but **no info logs** (errors only)
    if ($q = $db->prepare("SELECT alert_phone, alert_email FROM houses WHERE id = ? LIMIT 1")) {
        $q->bind_param("i", $house_id);
        // Execute silently; failure will just skip alerts (no logs unless you want errors here)
        @$q->execute();
        $q->close();
    }

    $db->close();

    echo json_encode([
        "status"         => "success",
        "message"        => "Visitor record saved.",
        "id"             => (int)$insertId,
        "image_metadata" => $image_metadata
    ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
    // Errors only
    error_log("visitor_post exception: " . $e->getMessage());
    json_error("Server error.", 500);
}
