<?php
// api/organisation/validate.php
require_once 'config.php';

header('Content-Type: application/json');
ob_start(); // Start output buffering

$conn = null; // Initialize connection variable

// Register shutdown function to ensure DB connection is closed
register_shutdown_function(function() use (&$conn) {
    if ($conn && $conn instanceof mysqli) {
        $conn->close();
        // Flush any remaining output buffer
        while (ob_get_level() > 0) {
            ob_end_flush();
        }
    }
});

try {
    $conn = getDbConnection(); // Get the database connection

    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    $contactEmail = isset($input['contact_email']) ? trim($input['contact_email']) : null;

    $exists = false;
    $message = 'Available';

    if ($contactEmail) {
        if (!filter_var($contactEmail, FILTER_VALIDATE_EMAIL)) {
            http_response_code(400); // Bad Request
            echo json_encode(['status' => 'error', 'message' => 'Invalid email format.']);
            exit(); // Exit immediately after sending response
        }

        $stmt = $conn->prepare("SELECT organisation_id FROM organisations WHERE contact_email = ? LIMIT 1");
        if (!$stmt) {
            throw new Exception("Prepare statement for organisation email check failed: " . $conn->error);
        }
        $stmt->bind_param("s", $contactEmail);
        $stmt->execute();
        $stmt->store_result();
        if ($stmt->num_rows > 0) {
            $exists = true;
            $message = 'Email is already used as an organization contact email.';
        }
        $stmt->close();

        if (!$exists) {
            $stmt = $conn->prepare("SELECT admin_id FROM admin WHERE username = ? LIMIT 1");
            if (!$stmt) {
                throw new Exception("Prepare statement for admin email check failed: " . $conn->error);
            }
            $stmt->bind_param("s", $contactEmail);
            $stmt->execute();
            $stmt->store_result();
            if ($stmt->num_rows > 0) {
                $exists = true;
                $message = 'Email is already registered as an admin username.';
            }
            $stmt->close();
        }
    } else {
        http_response_code(400); // Bad Request
        echo json_encode(['status' => 'error', 'message' => 'Missing contact_email parameter for validation.']);
        exit(); // Exit immediately after sending response
    }

    echo json_encode(['status' => 'success', 'message' => $message, 'exists' => $exists]);
    exit(); // Exit immediately after sending response

} catch (Exception $e) {
    http_response_code(500); // Internal Server Error
    error_log("Validation failed: " . $e->getMessage()); // Log internal error
    echo json_encode(['status' => 'error', 'message' => 'Validation failed: ' . $e->getMessage()]);
    exit(); // Exit immediately after sending response
}
// No closing PHP tag