<?php
// userdatafetch.php — fetch current user details by bearer token
// Returns: { status:"success", data:{ user_id, organisation_id, house_id, real_name, username } }

header("Content-Type: application/json; charset=utf-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

require_once __DIR__ . '/config.php';

function bearer_token(): ?string {
    $h = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['Authorization'] ?? '';
    if (!$h) return null;
    if (stripos($h, 'Bearer ') === 0) return trim(substr($h, 7));
    return trim($h); // accept raw token just in case
}
function json_error(string $msg, int $code = 400) {
    http_response_code($code);
    echo json_encode(["status" => "error", "message" => $msg], JSON_UNESCAPED_UNICODE);
    exit;
}

try {
    $token = bearer_token();
    if (!$token) json_error("Missing Authorization token.", 401);

    $db = getDbConnection();
    if (!$db) json_error("Database connection failed.", 500);
    $db->set_charset('utf8mb4');

    $sql = "SELECT user_id, organisation_id, house_id, real_name, username
            FROM users WHERE token = ? LIMIT 1";
    $st = $db->prepare($sql);
    if (!$st) json_error("Database error (prepare): ".$db->error, 500);
    $st->bind_param("s", $token);
    $st->execute();
    $res = $st->get_result();

    if ($res->num_rows === 0) json_error("Invalid or expired token.", 401);

    $row = $res->fetch_assoc();
    $st->close();
    $db->close();

    echo json_encode([
        "status" => "success",
        "data" => [
            "user_id" => (int)$row['user_id'],
            "organisation_id" => (int)$row['organisation_id'],
            "house_id" => isset($row['house_id']) ? (int)$row['house_id'] : null,
            "real_name" => (string)$row['real_name'],
            "username" => (string)$row['username']
        ]
    ], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
    json_error("Server error.", 500);
}
