<?php
// guardianapi/sms_stats.php
declare(strict_types=1);

header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

require_once __DIR__ . '/config.php';
$conn = getDbConnection();
if (!$conn) { http_response_code(500); echo json_encode(['status'=>'error','message'=>'DB unavailable']); exit; }

/* ------------ Auth (token) ------------ */
function gp_bearer_token(): ?string {
  $h = null;
  if (function_exists('getallheaders')) {
    $all = getallheaders();
    $h = $all['Authorization'] ?? $all['authorization'] ?? null;
  }
  if (!$h && isset($_SERVER['HTTP_AUTHORIZATION'])) $h = $_SERVER['HTTP_AUTHORIZATION'];
  if ($h && preg_match('/Bearer\s+([A-Za-z0-9._\-]+)/', $h, $m)) return $m[1];
  if (!empty($_GET['token'])) return (string)$_GET['token']; // legacy
  return null;
}
function gp_verify_token(mysqli $conn, ?string $token): ?array {
  if (!$token) return null;
  // admin
  if ($stmt = $conn->prepare("SELECT admin_id AS id, organisation_id FROM admin WHERE token=? AND is_active=1 LIMIT 1")) {
    $stmt->bind_param("s", $token); $stmt->execute(); $res = $stmt->get_result();
    if ($r = $res->fetch_assoc()) { $stmt->close(); return ['role'=>'admin','id'=>(int)$r['id'],'organisation_id'=>(int)$r['organisation_id'],'token'=>$token]; }
    $stmt->close();
  }
  // user
  if ($stmt = $conn->prepare("SELECT user_id AS id, organisation_id, house_id FROM users WHERE token=? LIMIT 1")) {
    $stmt->bind_param("s", $token); $stmt->execute(); $res = $stmt->get_result();
    if ($r = $res->fetch_assoc()) {
      $stmt->close();
      return ['role'=>'user','id'=>(int)$r['id'],'organisation_id'=>(int)$r['organisation_id'],'house_id'=> isset($r['house_id'])?(int)$r['house_id']:null,'token'=>$token];
    }
    $stmt->close();
  }
  return null;
}
$AUTH = $GLOBALS['AUTH'] ?? gp_verify_token($conn, gp_bearer_token());
if (!$AUTH) { http_response_code(401); echo json_encode(['status'=>'error','message'=>'Unauthorized']); exit; }

/* ------------ Inputs ------------ */
$paramOrg      = isset($_GET['organisation_id']) ? (int)$_GET['organisation_id'] : null;
$paramHouse    = isset($_GET['house_id']) ? (int)$_GET['house_id'] : null;
$from          = isset($_GET['from']) ? trim((string)$_GET['from']) : '';
$to            = isset($_GET['to']) ? trim((string)$_GET['to']) : '';
$includeFailed = isset($_GET['include_failed']) ? (int)$_GET['include_failed'] : 0;
$topLimit      = isset($_GET['limit']) ? max(1, min(200, (int)$_GET['limit'])) : 20;

/* ------------ Scope ------------ */
$isSuperAdmin = ($AUTH['role']==='admin' && (int)$AUTH['organisation_id']===7);
$effOrg   = $isSuperAdmin ? ($paramOrg && $paramOrg>0 ? $paramOrg : null) : (int)$AUTH['organisation_id'];
$effHouse = $paramHouse;
if ($AUTH['role']==='user' && array_key_exists('house_id',$AUTH) && $AUTH['house_id'] !== null) {
  $effHouse = (int)$AUTH['house_id'];
}

/* ------------ Helpers ------------ */
$tsExpr = "COALESCE(sm.sent_at, sm.created_at)";
function norm_dt(?string $s): ?string {
  $s = trim((string)$s);
  if ($s==='') return null;
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $s)) return $s.' 00:00:00';
  if (preg_match('/^\d{4}-\d{2}-\d{2}\s+\d{2}:\d{2}:\d{2}$/', $s)) return $s;
  return null;
}
function run_select(mysqli $conn, string $sql, string $types='', array $args=[]): array {
  $stmt = $conn->prepare($sql);
  if (!$stmt) return [];
  if ($types !== '' && $args) {
    $bind = [$types];
    foreach ($args as $k=>$v) $bind[] = &$args[$k];
    call_user_func_array([$stmt,'bind_param'],$bind);
  }
  $stmt->execute();
  $res = $stmt->get_result();
  $rows = [];
  if ($res) while ($row = $res->fetch_assoc()) $rows[] = $row;
  $stmt->close();
  return $rows;
}

/* ------------ WHERE (qualified with sm.) ------------ */
$statuses = $includeFailed ? ['sent','failed'] : ['sent'];

$conds = [];
$types = '';
$args  = [];

$in = implode(',', array_fill(0, count($statuses), '?'));
$conds[] = "sm.status IN ($in)";
$types  .= str_repeat('s', count($statuses));
foreach ($statuses as $s) $args[] = $s;

if ($effOrg !== null)   { $conds[] = "sm.organisation_id = ?"; $types .= 'i'; $args[] = $effOrg; }
if ($effHouse !== null) { $conds[] = "sm.house_id = ?";        $types .= 'i'; $args[] = $effHouse; }

$fromTs = norm_dt($from);
$toTs   = norm_dt($to);
if ($toTs && substr($toTs, 11)==='00:00:00') $toTs = substr($toTs, 0, 10).' 23:59:59';
if ($fromTs !== null) { $conds[] = "$tsExpr >= ?"; $types .= 's'; $args[] = $fromTs; }
if ($toTs   !== null) { $conds[] = "$tsExpr <= ?"; $types .= 's'; $args[] = $toTs; }

$where = $conds ? ('WHERE '.implode(' AND ', $conds)) : '';

/* ------------ Totals / Today / Month ------------ */
$totalRow = run_select($conn, "SELECT COUNT(*) AS c FROM sms_messages sm $where", $types, $args);
$totalFiltered = (int)($totalRow[0]['c'] ?? 0);

$todayStart = date('Y-m-d 00:00:00'); $todayEnd = date('Y-m-d 23:59:59');
$tc = $conds; $tt = $types; $ta = $args;
// strip any existing time bounds (those containing $tsExpr)
$tc = array_values(array_filter($tc, fn($c) => strpos($c, 'COALESCE(sm.sent_at, sm.created_at)') === false));
$tc[] = "$tsExpr >= ?"; $tt .= 's'; $ta[] = $todayStart;
$tc[] = "$tsExpr <= ?"; $tt .= 's'; $ta[] = $todayEnd;
$tw = 'WHERE '.implode(' AND ', $tc);
$todayRow = run_select($conn, "SELECT COUNT(*) AS c FROM sms_messages sm $tw", $tt, $ta);
$todayCount = (int)($todayRow[0]['c'] ?? 0);

$monthStart = date('Y-m-01 00:00:00'); $monthEnd = date('Y-m-t 23:59:59');
$mc = $conds; $mt = $types; $ma = $args;
$mc = array_values(array_filter($mc, fn($c) => strpos($c, 'COALESCE(sm.sent_at, sm.created_at)') === false));
$mc[] = "$tsExpr >= ?"; $mt .= 's'; $ma[] = $monthStart;
$mc[] = "$tsExpr <= ?"; $mt .= 's'; $ma[] = $monthEnd;
$mw = 'WHERE '.implode(' AND ', $mc);
$monthRow = run_select($conn, "SELECT COUNT(*) AS c FROM sms_messages sm $mw", $mt, $ma);
$monthCount = (int)($monthRow[0]['c'] ?? 0);

/* ------------ Series & Top lists ------------ */
$seriesRows = run_select(
  $conn,
  "SELECT DATE($tsExpr) AS d, COUNT(*) AS c
     FROM sms_messages sm
     $where
     GROUP BY DATE($tsExpr)
     ORDER BY DATE($tsExpr) ASC",
  $types, $args
);
$daily = [];
foreach ($seriesRows as $r) $daily[] = ['date'=>(string)$r['d'], 'count'=>(int)$r['c']];

$byOrgRows = run_select(
  $conn,
  "SELECT sm.organisation_id, COUNT(*) AS c
     FROM sms_messages sm
     $where
     GROUP BY sm.organisation_id
     ORDER BY c DESC
     LIMIT $topLimit",
  $types, $args
);
$by_org = array_map(fn($r)=>['organisation_id'=>(int)$r['organisation_id'], 'count'=>(int)$r['c']], $byOrgRows);

$byHouseRows = run_select(
  $conn,
  "SELECT sm.house_id, COALESCE(h.house_number,'') AS house_number, COUNT(*) AS c
     FROM sms_messages sm
     LEFT JOIN houses h ON h.id = sm.house_id
     $where
     GROUP BY sm.house_id, house_number
     ORDER BY c DESC
     LIMIT $topLimit",
  $types, $args
);
$by_house = array_map(fn($r)=>[
  'house_id'=> isset($r['house_id']) ? (int)$r['house_id'] : null,
  'house_number'=> (string)$r['house_number'],
  'count'=> (int)$r['c']
], $byHouseRows);

$most_active_org   = $by_org[0]   ?? null;
$most_active_house = $by_house[0] ?? null;

/* ------------ Response ------------ */
echo json_encode([
  'status'  => 'success',
  'scope'   => [
    'actor_role'       => $AUTH['role'],
    'actor_org'        => (int)$AUTH['organisation_id'],
    'effective_org'    => $effOrg,    // null = all (superadmin)
    'effective_house'  => $effHouse,
  ],
  'filters' => [
    'from' => $fromTs, 'to' => $toTs,
    'include_failed' => $includeFailed,
  ],
  'totals' => [
    'filtered' => $totalFiltered,
    'today'    => $todayCount,
    'month'    => $monthCount,
  ],
  'daily'    => $daily,
  'by_org'   => $by_org,
  'by_house' => $by_house,
  'most_active_org'   => $most_active_org,
  'most_active_house' => $most_active_house,
], JSON_UNESCAPED_UNICODE);
exit;
