<?php
// guardianapi/sms_quota_set.php
declare(strict_types=1);

header('Content-Type: application/json; charset=UTF-8');

if (!isset($GLOBALS['AUTH'])) {
  http_response_code(401);
  echo json_encode(['status'=>'error','message'=>'Unauthorized']);
  exit;
}
$auth = $GLOBALS['AUTH'];

require_once __DIR__ . '/config.php';
$conn = getDbConnection();

/* -------- helpers (avoid name clashes with router) -------- */
if (!function_exists('gp_respond')) {
  function gp_respond(array $arr, int $code=200){ http_response_code($code); echo json_encode($arr); exit; }
}
if (!function_exists('gp_int_or_null')) {
  function gp_int_or_null($v){ return ($v === '' || $v === null || !isset($v)) ? null : (int)$v; }
}

/* -------- permission model -------- */
$role     = (string)($auth['role'] ?? '');
$actorOrg = (int)($auth['organisation_id'] ?? -1);

if ($role !== 'admin') gp_respond(['status'=>'error','message'=>'Forbidden'], 403);

// Superadmin set: org 0, org 7, and optional SUPERADMIN_ORG_ID
$superIds = [0, 7];
if (defined('SUPERADMIN_ORG_ID')) $superIds[] = (int)SUPERADMIN_ORG_ID;
$superIds = array_values(array_unique($superIds));
$isSuper  = in_array($actorOrg, $superIds, true);

/* -------- input -------- */
$in = json_decode(file_get_contents('php://input'), true) ?: [];

$orgId   = isset($in['organisation_id']) ? (int)$in['organisation_id'] : 0;
$houseId = array_key_exists('house_id', $in) ? gp_int_or_null($in['house_id']) : null;
$perDay  = array_key_exists('per_day',   $in) ? gp_int_or_null($in['per_day'])   : null;
$perMon  = array_key_exists('per_month', $in) ? gp_int_or_null($in['per_month']) : null;

// Default org: if not provided, use caller's org (works for non-super admins)
if ($orgId <= 0) $orgId = $actorOrg;

// Enforce scope: superadmin can set any; normal admin only own org
if (!$isSuper && $orgId !== $actorOrg) {
  gp_respond(['status'=>'error','message'=>'Cannot modify another organisation'], 403);
}
if ($orgId <= 0) {
  gp_respond(['status'=>'error','message'=>'Missing organisation_id'], 400);
}

// Normalize limits: negative/zero -> deactivate (treat as null)
if ($perDay !== null && $perDay <= 0) $perDay = null;
if ($perMon !== null && $perMon <= 0) $perMon = null;

/* -------- DB logic -------- */
$conn->begin_transaction();

function upsert_limit(mysqli $c, int $orgId, ?int $houseId, string $period, ?int $limit): void {
  if ($limit === null) {
    // deactivate existing rows for this scope/period
    $sql = "UPDATE sms_quotas
               SET is_active=0
             WHERE organisation_id=?
               AND ((? IS NULL AND house_id IS NULL) OR house_id=?)
               AND period=?";
    $st = $c->prepare($sql);
    $st->bind_param("iiis", $orgId, $houseId, $houseId, $period);
    $st->execute(); $st->close();
    return;
  }

  // try update first
  $sql = "UPDATE sms_quotas
             SET limit_count=?, is_active=1
           WHERE organisation_id=?
             AND ((? IS NULL AND house_id IS NULL) OR house_id=?)
             AND period=?";
  $st = $c->prepare($sql);
  $st->bind_param("iiiss", $limit, $orgId, $houseId, $houseId, $period);
  $st->execute();
  $aff = $st->affected_rows;
  $st->close();
  if ($aff > 0) return;

  // insert new row for this scope/period
  if ($houseId === null) {
    $sql = "INSERT INTO sms_quotas (organisation_id, house_id, period, limit_count, is_active)
            VALUES (?, NULL, ?, ?, 1)";
    $st = $c->prepare($sql);
    $st->bind_param("isi", $orgId, $period, $limit);
    $st->execute(); $st->close();
  } else {
    $sql = "INSERT INTO sms_quotas (organisation_id, house_id, period, limit_count, is_active)
            VALUES (?, ?, ?, ?, 1)";
    $st = $c->prepare($sql);
    $st->bind_param("iisi", $orgId, $houseId, $period, $limit);
    $st->execute(); $st->close();
  }
}

try {
  if (array_key_exists('per_day', $in))   upsert_limit($conn, $orgId, $houseId, 'daily',   $perDay);
  if (array_key_exists('per_month', $in)) upsert_limit($conn, $orgId, $houseId, 'monthly', $perMon);
  $conn->commit();
} catch (Throwable $e) {
  $conn->rollback();
  gp_respond(['status'=>'error','message'=>'DB error: '.$e->getMessage()], 500);
}

gp_respond(['status'=>'success','message'=>'Quota updated']);
