<?php
// guardianapi/registerorg.php
declare(strict_types=1);

header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// Preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

require_once __DIR__ . '/config.php';

// PHPMailer (match settings used in send_visitor_alert.php)
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
require_once __DIR__ . '/src/Exception.php';
require_once __DIR__ . '/src/PHPMailer.php';
require_once __DIR__ . '/src/SMTP.php';

// --- helpers (use unique names to avoid collisions) ---
if (!function_exists('gp_log')) {
  function gp_log(string $msg): void {
    @file_put_contents(__DIR__ . '/email_debug.log',
      date('Y-m-d H:i:s') . ' - ' . $msg . PHP_EOL, FILE_APPEND);
  }
}

if (!function_exists('gp_out')) {
  function gp_out(array $arr, int $code = 200): void {
    http_response_code($code);
    echo json_encode($arr);
    exit;
  }
}

/**
 * Send internal notification email to GuestPass onboarding inbox.
 */
function gp_send_lead_email(array $lead): bool {
  $mail = new PHPMailer(true);
  try {
    $orgName  = $lead['organisation_name'];
    $ctName   = $lead['contact_name'];
    $ctEmail  = $lead['contact_email'];
    $ctPhone  = $lead['contact_phone'];
    $now      = date('Y-m-d H:i:s');
    $ip       = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $ua       = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';

    $html  = "<h2>New Organisation Registration Request</h2>";
    $html .= "<p><strong>When:</strong> {$now}</p>";
    $html .= "<p><strong>IP:</strong> {$ip}<br><strong>User-Agent:</strong> {$ua}</p>";
    $html .= "<hr>";
    $html .= "<p><strong>Organisation:</strong> " . htmlspecialchars($orgName) . "</p>";
    $html .= "<p><strong>Contact Name:</strong> " . htmlspecialchars($ctName)  . "</p>";
    $html .= "<p><strong>Contact Email:</strong> " . htmlspecialchars($ctEmail) . "</p>";
    $html .= "<p><strong>Contact Phone:</strong> " . htmlspecialchars($ctPhone) . "</p>";
    if (!empty($lead['admin_username'])) {
      $html .= "<p><strong>Requested Admin Username:</strong> " . htmlspecialchars($lead['admin_username']) . "</p>";
    }
    if (!empty($lead['notes'])) {
      $html .= "<p><strong>Notes:</strong><br>" . nl2br(htmlspecialchars($lead['notes'])) . "</p>";
    }

    // SMTP (same as send_visitor_alert.php)
    $mail->isSMTP();
    $mail->Host       = 'mail.parkspace.co.ke';
    $mail->SMTPAuth   = true;
    $mail->Username   = 'guestpass@parkspace.co.ke';
    $mail->Password   = 'Shemking@18';
    $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS; // SMTPS (465)
    $mail->Port       = 465;

    $mail->setFrom('guestpass@parkspace.co.ke', 'GuestPass VMS');
    $mail->addAddress('connect@guestpassvms.com', 'GuestPass Onboarding');
    // Extra inbox (as requested)
    $mail->addCC('shemvilejado@gmail.com');

    // So you can reply straight to the requester
    if (filter_var($ctEmail, FILTER_VALIDATE_EMAIL)) {
      $mail->addReplyTo($ctEmail, $ctName ?: $ctEmail);
    }

    $mail->isHTML(true);
    $mail->Subject = 'New Organisation Registration Request' . $orgName;
    $mail->Body    = $html;

    $mail->send();
    gp_log("Lead email sent for org '{$orgName}'");
    return true;
  } catch (Exception $e) {
    gp_log("Lead email FAILED: " . $e->getMessage());
    return false;
  }
}

/**
 * Send you an SMS ping via the configured provider keys in config.php.
 * Does NOT expose credentials to the client.
 */
function gp_sms_owner_ping(string $text): array {
  // Use the same constants as sms_send.php
  $short = defined('SMS_SHORTCODE') ? SMS_SHORTCODE : 'Signal VMS';
  $payload = [
    'apikey'    => defined('SMS_APIKEY')  ? SMS_APIKEY  : '',
    'partnerID' => defined('SMS_PARTNER') ? SMS_PARTNER : '',
    'message'   => $text,
    'shortcode' => $short,
    'mobile'    => '254742835803',
  ];

  $url = (defined('SMS_BASE') ? SMS_BASE : '') . 'sendsms/';
  if (!$url) {
    gp_log('SMS ping aborted: SMS_BASE not configured');
    return ['ok'=>false, 'code'=>0, 'desc'=>'SMS_BASE not configured'];
  }

  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
    CURLOPT_POST           => true,
    CURLOPT_POSTFIELDS     => json_encode($payload, JSON_UNESCAPED_UNICODE),
    CURLOPT_TIMEOUT        => 20,
  ]);
  $resp = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($err) {
    gp_log("SMS ping transport error: {$err}");
    return ['ok'=>false, 'code'=>$code, 'desc'=>$err];
  }

  $json = json_decode($resp, true);
  $r0   = $json['responses'][0] ?? null;
  $pcode= (int)($r0['respose-code'] ?? $r0['response-code'] ?? 0);
  $desc = (string)($r0['response-description'] ?? '');
  $ok   = ($pcode === 200);

  gp_log("SMS ping result code={$pcode} desc={$desc}");
  return ['ok'=>$ok, 'code'=>$pcode, 'desc'=>$desc];
}

// --- main ---
try {
  $in = json_decode(file_get_contents('php://input'), true) ?: [];

  // Keep previous input shape for compatibility
  $orgName  = trim((string)($in['organisation_name'] ?? ($in['name'] ?? '')));
  $ctName   = trim((string)($in['contact_name']     ?? ''));
  $ctEmail  = trim((string)($in['contact_email']    ?? ($in['admin_email'] ?? '')));
  $ctPhone  = trim((string)($in['contact_phone']    ?? ($in['admin_phone'] ?? '')));
  $admUser  = trim((string)($in['admin_username']   ?? $ctEmail));
  $admPass  = (string)($in['admin_password']        ?? '');  // not used now, but accepted
  $notes    = trim((string)($in['notes']            ?? ''));

  if ($orgName === '' || $ctEmail === '' || $ctPhone === '' || $ctName === '' || !filter_var($ctEmail, FILTER_VALIDATE_EMAIL)) {
    gp_out(['status'=>'error','message'=>'Missing or invalid required fields (organisation_name, contact_name, contact_email, contact_phone)'], 400);
  }

  // We no longer auto-create. Instead we notify the onboarding inbox and send you an SMS ping.
  $lead = [
    'organisation_name' => $orgName,
    'contact_name'      => $ctName,
    'contact_email'     => $ctEmail,
    'contact_phone'     => $ctPhone,
    'admin_username'    => $admUser,
    'notes'             => $notes,
  ];

  $emailOk = gp_send_lead_email($lead);

  $smsText = "GuestPass lead: {$orgName} — {$ctName} {$ctPhone}";
  $smsRes  = gp_sms_owner_ping($smsText);

  // Always return success to the app unless both failed hard
  if (!$emailOk && !$smsRes['ok']) {
    gp_out(['status'=>'error','message'=>'Notification failed; please try again later'], 500);
  }

  gp_out([
    'status'   => 'success',
    'message'  => 'Thanks! We have received your request and will contact you.',
    'email_ok' => $emailOk,
    'sms_ok'   => $smsRes['ok'],
  ]);

} catch (Throwable $e) {
  gp_log('registerorg fatal: ' . $e->getMessage());
  gp_out(['status'=>'error','message'=>'Server error'], 500);
}