<?php
/**
 * guardianapi/house_post.php
 * Upsert into `houses` using (organisation_id, house_number).
 * Fully compatible with your existing config.php (no changes needed).
 * `alert_email` is optional; `alert_phone` or legacy `alert_numbers` can be provided.
 */

declare(strict_types=1);

require_once __DIR__ . '/config.php';

/* -------- Error logging -------- */
ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/errors.log');

/* -------- CORS / preflight -------- */
gp_set_cors_all();
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  http_response_code(204);
  exit;
}
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  gp_respond(['status' => 'error', 'message' => 'Method not allowed'], 405);
}

try {
  $data = gp_json_input();

  // Back-compat: map alert_numbers → alert_phone
  if (isset($data['alert_numbers']) && empty($data['alert_phone'])) {
    $data['alert_phone'] = $data['alert_numbers'];
  }

  // --- Required fields ---
  if (!isset($data['organisation_id']) || !is_numeric($data['organisation_id'])) {
    throw new Exception('Missing or invalid organisation_id');
  }
  if (!isset($data['house_number']) || trim((string)$data['house_number']) === '') {
    throw new Exception('Missing or invalid house_number');
  }

  $organisation_id = (int)$data['organisation_id'];
  $house_number    = trim((string)$data['house_number']);
  $alert_email     = isset($data['alert_email']) ? trim((string)$data['alert_email']) : null;
  $alert_phone     = isset($data['alert_phone']) ? trim((string)$data['alert_phone']) : null;

  // Sanitize lengths
  if (strlen($house_number) > 100) $house_number = substr($house_number, 0, 100);
  if ($alert_email && strlen($alert_email) > 255) $alert_email = substr($alert_email, 0, 255);
  if ($alert_phone && strlen($alert_phone) > 20) $alert_phone = substr($alert_phone, 0, 20);

  $conn = getDbConnection();

  // --- Check if exists ---
  $check = $conn->prepare("SELECT id FROM houses WHERE organisation_id = ? AND house_number = ? LIMIT 1");
  if (!$check) throw new Exception('Prepare failed: ' . $conn->error);
  $check->bind_param("is", $organisation_id, $house_number);
  $check->execute();
  $check->bind_result($existing_id);
  $found = $check->fetch();
  $check->close();

  if ($found) {
    // --- UPDATE ---
    $set = "updated_at = NOW()";
    $params = [];
    $types  = '';

    if ($alert_email !== null) {
      $set .= ", alert_email = ?";
      $params[] = $alert_email;
      $types   .= 's';
    }
    if ($alert_phone !== null) {
      $set .= ", alert_phone = ?";
      $params[] = $alert_phone;
      $types   .= 's';
    }

    $sql = "UPDATE houses SET $set WHERE id = ?";
    $params[] = (int)$existing_id;
    $types   .= 'i';

    $stmt = $conn->prepare($sql);
    if (!$stmt) throw new Exception('Prepare failed: ' . $conn->error);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $stmt->close();

    gp_respond([
      'status'  => 'success',
      'message' => 'House updated successfully',
      'data'    => ['id' => (int)$existing_id, 'mode' => 'update']
    ], 200);

  } else {
    // --- INSERT ---
    $sql = "INSERT INTO houses (organisation_id, house_number, alert_email, alert_phone, created_at, updated_at)
            VALUES (?, ?, ?, ?, NOW(), NOW())";
    $stmt = $conn->prepare($sql);
    if (!$stmt) throw new Exception('Prepare failed: ' . $conn->error);
    $stmt->bind_param("isss", $organisation_id, $house_number, $alert_email, $alert_phone);
    $stmt->execute();
    $new_id = $stmt->insert_id;
    $stmt->close();

    gp_respond([
      'status'  => 'success',
      'message' => 'House registered successfully',
      'data'    => ['id' => (int)$new_id, 'mode' => 'insert']
    ], 201);
  }

} catch (Throwable $e) {
  error_log('house_post.php error: ' . $e->getMessage());
  gp_respond(['status' => 'error', 'message' => $e->getMessage()], 400);
}
