<?php
// guardianapi/house_get.php
declare(strict_types=1);
header("Content-Type: application/json; charset=utf-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET");

require_once __DIR__ . '/config.php';
$conn = getDbConnection(); // mysqli
$conn->set_charset('utf8mb4');

/* --------------------------- Helpers --------------------------- */
function hg_error(string $msg, int $code = 400): void {
  http_response_code($code);
  echo json_encode(["status" => "error", "message" => $msg]);
  exit;
}
function hg_norm_bool($v): ?int {
  if ($v === null) return null;
  if (is_bool($v)) return $v ? 1 : 0;
  if (is_numeric($v)) return ((int)$v) > 0 ? 1 : 0;
  $s = strtolower((string)$v);
  if (in_array($s, ['1','true','yes','active'], true)) return 1;
  if (in_array($s, ['0','false','no','inactive'], true)) return 0;
  return null;
}

/* --------------------------- Input ----------------------------- */
if (!isset($_GET['organisation_id'])) hg_error("Missing organisation_id parameter.");
$orgId = (int) $_GET['organisation_id'];
if ($orgId <= 0) hg_error("Invalid organisation_id.");

$houseNumber = isset($_GET['house_number']) ? trim((string)$_GET['house_number']) : '';

$active    = isset($_GET['active'])    ? hg_norm_bool($_GET['active'])    : null;
$effective = isset($_GET['effective']) ? hg_norm_bool($_GET['effective']) : null;
$expired   = isset($_GET['expired'])   ? hg_norm_bool($_GET['expired'])   : null;

if ($effective !== null && $expired !== null) $expired = null; // effective wins

/* -------------------------- Query ------------------------------ */
$sql = "
  SELECT 
    id,
    house_number,
    alert_email,
    COALESCE(alert_phone, '') AS alert_phone,
    COALESCE(alert_phone, '') AS alert_no,
    is_active,
    active_until,
    status,
    CASE 
      WHEN status = 'Active' AND is_active = 1 AND (active_until IS NULL OR active_until >= NOW()) THEN 1
      ELSE 0
    END AS effective_active
  FROM houses
  WHERE organisation_id = ?
    AND status = 'Active'  -- hide soft-deleted by default
";

$conds = [];
$params = [$orgId];
$types  = "i";

if ($houseNumber !== '') {
  $conds[] = "house_number = ?";
  $types  .= "s";
  $params[] = $houseNumber;
}
if ($active !== null) {
  $conds[] = "is_active = ?";
  $types  .= "i";
  $params[] = $active;
}
if ($effective !== null) {
  if ($effective === 1) {
    $conds[] = "(is_active = 1 AND (active_until IS NULL OR active_until >= NOW()))";
  } else {
    $conds[] = "(is_active = 0 OR (active_until IS NOT NULL AND active_until < NOW()))";
  }
}
if ($expired !== null) {
  if ($expired === 1) $conds[] = "(active_until IS NOT NULL AND active_until < NOW())";
  else                $conds[] = "(active_until IS NULL OR active_until >= NOW())";
}

if (!empty($conds)) $sql .= " AND " . implode(" AND ", $conds);
$sql .= " ORDER BY house_number ASC";

$stmt = $conn->prepare($sql);
if (!$stmt) hg_error("Database error: " . $conn->error, 500);

$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();

$houses = [];
while ($row = $result->fetch_assoc()) {
  $row['alert_phone']      = trim((string)$row['alert_phone']);
  $row['alert_no']         = trim((string)$row['alert_no']);
  $row['is_active']        = (int)$row['is_active'];
  $row['effective_active'] = (int)$row['effective_active'];
  $houses[] = $row;
}

echo json_encode(empty($houses)
  ? ["status" => "error", "message" => "No houses found for the given filters."]
  : ["status" => "success", "houses" => $houses]
);

$stmt->close();
$conn->close();
