<?php
header('Content-Type: application/json');

// 1) Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Use POST'
    ]);
    exit;
}

// 2) Read & decode JSON
$body = file_get_contents('php://input');
$data = json_decode($body, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Malformed JSON'
    ]);
    exit;
}

// 3) Validate parameters
$user = isset($data['user_id'])   ? (int)$data['user_id']   : 0;
$site = isset($data['site_id'])   ? (int)$data['site_id']   : 0;
$tag  = isset($data['tag'])       ? trim($data['tag'])      : '';

if ($user <= 0 || $site <= 0 || $tag === '') {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Missing or invalid parameters'
    ]);
    exit;
}

// Cap tag length
$tag = substr($tag, 0, 64);

try {
    // 4) Obtain PDO & start transaction
    require 'config.php';
    $pdo = getPdoConnection();
    $pdo->beginTransaction();

    // 5) Find the open session
    $sql  = "SELECT id
               FROM book_sessions
              WHERE user_id = :u
                AND site_id = :s
                AND clock_out_time IS NULL
              ORDER BY clock_in_time DESC
              LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(['u' => $user, 's' => $site]);
    $row  = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        $pdo->rollBack();
        echo json_encode([
            'success' => false,
            'message' => 'No active session'
        ]);
        exit;
    }

    // 6) Update the record
    $sql = "UPDATE book_sessions
               SET clock_out_tag  = :t,
                   clock_out_time = NOW()
             WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        't'  => $tag,
        'id' => $row['id']
    ]);

    // 7) Confirm update
    if ($stmt->rowCount() !== 1) {
        $pdo->rollBack();
        echo json_encode([
            'success' => false,
            'message' => 'Session already closed'
        ]);
        exit;
    }

    // 8) Commit & respond
    $pdo->commit();
    echo json_encode([
        'success' => true,
        'message' => 'Clock-out recorded'
    ]);
    exit;

} catch (PDOException $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Database error',
        'error'   => $e->getMessage()
    ]);
    exit;
}
